/*
 * Copyright (C) 2000 Rich Wareham <richwareham@users.sourceforge.net>
 *
 * This file is part of libdvdnav, a DVD navigation library.
 *
 * libdvdnav is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libdvdnav is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with libdvdnav; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* Un-comment the following to enable additional Log3() function calls.
 * Do not forget to #define TRACE in src/vm/vm.h (required) */
/*
#define LOG_DEBUG
*/

#include <inttypes.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <limits.h>
#include <string.h>
#include "dvdnav/dvdnav.h"
#include <dvdread/dvd_reader.h>
#include <dvdread/nav_types.h>
#include "vm/decoder.h"
#include "vm/vm.h"
#include "vm/getset.h"
#include "dvdnav_internal.h"
#include "logger.h"
#ifndef DVDNAV_NO_CACHE
#include "read_cache.h"
#endif
#include <dvdread/nav_read.h>

static dvdnav_status_t dvdnav_clear(dvdnav_t * this) {
  /* clear everything except file, vm, mutex, readahead */

  pthread_mutex_lock(&this->vm_lock);
  if (this->file) DVDCloseFile(this->file);
  this->file = NULL;

  memset(&this->position_current,0,sizeof(this->position_current));
  memset(&this->pci,0,sizeof(this->pci));
  memset(&this->dsi,0,sizeof(this->dsi));
  this->last_cmd_nav_lbn = SRI_END_OF_CELL;

  /* Set initial values of flags */
  this->skip_still = 0;
  this->sync_wait = 0;
  this->sync_wait_skip = 0;
  this->spu_clut_changed = 0;
  this->started = 0;
  this->nav_only = 0;
  this->cur_cell_time = 0;

#ifndef DVDNAV_NO_CACHE
  dvdnav_read_cache_clear(this->cache);
#endif
  pthread_mutex_unlock(&this->vm_lock);

  return DVDNAV_STATUS_OK;
}

dvdnav_status_t dvdnav_dup(dvdnav_t **dest, dvdnav_t *src) {
  dvdnav_t *this;

  (*dest) = NULL;
  this = (dvdnav_t*)malloc(sizeof(dvdnav_t));
  if (!this)
    return DVDNAV_STATUS_ERR;

  memcpy(this, src, sizeof(dvdnav_t));
  this->file = NULL;
  this->vm = NULL;
  this->path = NULL;
#ifndef DVDNAV_NO_CACHE
  this->cache = NULL;
#endif

  pthread_mutex_init(&this->vm_lock, NULL);

  this->vm = vm_new_copy(src->vm);
  if (!this->vm)
    goto fail;

  if (src->path) {
      this->path = strdup(src->path);
      if (!this->path)
          goto fail;
  }

#ifndef DVDNAV_NO_CACHE
  /* Start the read-ahead cache. */
  this->cache = dvdnav_read_cache_new(this);
  if (!this->cache)
    goto fail;
#endif

  (*dest) = this;
  return DVDNAV_STATUS_OK;

fail:
    printerr("Error initialising the DVD VM.");
    pthread_mutex_destroy(&this->vm_lock);
    vm_free_vm(this->vm);
    free(this->path);
    free(this);
    return DVDNAV_STATUS_ERR;
}

dvdnav_status_t dvdnav_free_dup(dvdnav_t *this) {

#ifdef LOG_DEBUG
  Log3(this, "free_dup:called");
#endif

  if (this->file) {
    pthread_mutex_lock(&this->vm_lock);
    DVDCloseFile(this->file);
#ifdef LOG_DEBUG
    Log3(this, "close:file closing");
#endif
    this->file = NULL;
    pthread_mutex_unlock(&this->vm_lock);
  }

  /* Free the VM */
  if(this->vm)
    vm_free_copy(this->vm);

  pthread_mutex_destroy(&this->vm_lock);

  free(this->path);

#ifndef DVDNAV_NO_CACHE
  /* We leave the final freeing of the entire structure to the cache,
   * because we don't know, if there are still buffers out in the wild,
   * that must return first. */
  if(this->cache)
    dvdnav_read_cache_free(this->cache);
  else
    free(this);
#else
  free(this);
#endif

  return DVDNAV_STATUS_OK;
}

static dvdnav_status_t dvdnav_open_common(dvdnav_t** dest,
                                          void *priv, const dvdnav_logger_cb *logcb,
                                          const char *path,
                                          dvdnav_stream_cb *stream_cb) {
  dvdnav_t *this;

  /* Create a new structure */
  (*dest) = NULL;
  this = (dvdnav_t*)calloc(1, sizeof(dvdnav_t));
  if(!this)
    return DVDNAV_STATUS_ERR;

  this->priv = priv;
  if(logcb)
    this->logcb = *logcb;

  pthread_mutex_init(&this->vm_lock, NULL);
  /* Initialise the error string */
  printerr("");

  /* Initialise the VM */
  this->vm = vm_new_vm(priv, logcb);
  if(!this->vm) {
    goto fail;
  }
  if(!vm_reset(this->vm, path, priv, stream_cb)) {
    goto fail;
  }

  /* Set the path. */
  if(path != NULL)
  {
    this->path = strdup(path);
    if(!this->path)
      goto fail;
  }

  /* Pre-open and close a file so that the CSS-keys are cached. */
  this->file = DVDOpenFile(vm_get_dvd_reader(this->vm), 0, DVD_READ_MENU_VOBS);

#ifndef DVDNAV_NO_CACHE
  /* Start the read-ahead cache. */
  this->cache = dvdnav_read_cache_new(this);
  if(!this->cache)
    goto fail;
#endif

  dvdnav_clear(this);

  (*dest) = this;
  return DVDNAV_STATUS_OK;

fail:
  pthread_mutex_destroy(&this->vm_lock);
  vm_free_vm(this->vm);
  free(this->path);
  free(this);
  return DVDNAV_STATUS_ERR;
}

dvdnav_status_t dvdnav_open(dvdnav_t** dest, const char *path) {
  return dvdnav_open_common(dest, NULL, NULL, path, NULL);
}

dvdnav_status_t dvdnav_open2(dvdnav_t** dest,
                             void *priv,const dvdnav_logger_cb *logcb,
                             const char *path) {
  return dvdnav_open_common(dest, priv, logcb, path, NULL);
}

dvdnav_status_t dvdnav_open_stream(dvdnav_t** dest,
                                   void *priv, dvdnav_stream_cb *stream_cb) {
  return dvdnav_open_common(dest, priv, NULL, NULL, stream_cb);
}

dvdnav_status_t dvdnav_open_stream2(dvdnav_t** dest,
                                    void *priv,const dvdnav_logger_cb *logcb,
                                    dvdnav_stream_cb *stream_cb) {
  return dvdnav_open_common(dest, priv, logcb, NULL, stream_cb);
}

dvdnav_status_t dvdnav_close(dvdnav_t *this) {

#ifdef LOG_DEBUG
  Log3(this, "close:called");
#endif

  if (this->file) {
    pthread_mutex_lock(&this->vm_lock);
    DVDCloseFile(this->file);
#ifdef LOG_DEBUG
    Log3(this, "close:file closing");
#endif
    this->file = NULL;
    pthread_mutex_unlock(&this->vm_lock);
  }

  /* Free the VM */
  if(this->vm)
    vm_free_vm(this->vm);

  pthread_mutex_destroy(&this->vm_lock);

  free(this->path);

#ifndef DVDNAV_NO_CACHE
  /* We leave the final freeing of the entire structure to the cache,
   * because we don't know, if there are still buffers out in the wild,
   * that must return first. */
  if(this->cache)
    dvdnav_read_cache_free(this->cache);
  else
    free(this);
#else
  free(this);
#endif

  return DVDNAV_STATUS_OK;
}

dvdnav_status_t dvdnav_reset(dvdnav_t *this) {
  dvdnav_status_t result;

#ifdef LOG_DEBUG
  Log3(this, "reset:called");
#endif

  pthread_mutex_lock(&this->vm_lock);

#ifdef LOG_DEBUG
  Log3(this, "resetting vm");
#endif
  if(!vm_reset(this->vm, NULL, NULL, NULL)) {
    printerr("Error restarting the VM.");
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_ERR;
  }
#ifdef LOG_DEBUG
  Log3(this, "clearing dvdnav");
#endif
  pthread_mutex_unlock(&this->vm_lock);
  result = dvdnav_clear(this);

  return result;
}

dvdnav_status_t dvdnav_path(dvdnav_t *this, const char** path) {
  (*path) = this->path;

  return DVDNAV_STATUS_OK;
}

const char* dvdnav_err_to_string(dvdnav_t *this) {

#ifndef DVDNAV_NO_ERRSTR
    if(!this)
    return "Hey! You gave me a NULL pointer you naughty person!";

  return this->err_str;
#else
    return "";
#endif

}

/* converts a dvd_time_t to PTS ticks */
int64_t dvdnav_convert_time(const dvd_time_t *time) {
  int64_t result;
  int64_t frames;

  result  = ((int64_t)(time->hour    >> 4  )) * 10 * 60 * 60 * 90000;
  result += ((int64_t)(time->hour    & 0x0f))      * 60 * 60 * 90000;
  result += ((int64_t)(time->minute  >> 4  ))      * 10 * 60 * 90000;
  result += ((int64_t)(time->minute  & 0x0f))           * 60 * 90000;
  result += ((int64_t)(time->second  >> 4  ))           * 10 * 90000;
  result += ((int64_t)(time->second  & 0x0f))                * 90000;

  frames  = ((time->frame_u & 0x30) >> 4) * 10;
  frames += ((time->frame_u & 0x0f)     )     ;

  if (time->frame_u & 0x80)
    result += frames * 3000;
  else
    result += frames * 3600;

  return result;
}

/*
 * Returns 1 if block contains NAV packet, 0 otherwise.
 * Processes said NAV packet if present.
 *
 * Most of the code in here is copied from xine's MPEG demuxer
 * so any bugs which are found in that should be corrected here also.
 */
static int32_t dvdnav_decode_packet(dvdnav_t *this, uint8_t *p,
                                    dsi_t *nav_dsi, pci_t *nav_pci) {
  int32_t        bMpeg1 = 0;
  uint32_t       nHeaderLen;
  uint32_t       nPacketLen;
  uint32_t       nStreamID;

  if (p[3] == 0xBA) { /* program stream pack header */
    int32_t nStuffingBytes;

    bMpeg1 = (p[4] & 0x40) == 0;

    if (bMpeg1) {
      p += 12;
    } else { /* mpeg2 */
      nStuffingBytes = p[0xD] & 0x07;
      p += 14 + nStuffingBytes;
    }
  }

  if (p[3] == 0xbb) { /* program stream system header */
    nHeaderLen = (p[4] << 8) | p[5];
    p += 6 + nHeaderLen;
  }

  /* we should now have a PES packet here */
  if (p[0] || p[1] || (p[2] != 1)) {
    Log1(this, "demux error! %02x %02x %02x (should be 0x000001)",p[0],p[1],p[2]);
    return 0;
  }

  nPacketLen = p[4] << 8 | p[5];
  nStreamID  = p[3];

  nHeaderLen = 6;
  p += nHeaderLen;

  if (nStreamID == 0xbf) { /* Private stream 2 */
#if 0
    char buffer[80 * 3 + 1];
    int32_t i;
    for(i=0;i<80;i++)
      sprintf(&buffer[i*3], "%02x ",p[i-6]);
    Log3(this, "nav packet=%u %s",p-p_start-6, buffer);
#endif

    if(p[0] == 0x00) {
      navRead_PCI(nav_pci, p+1);
    }

    p += nPacketLen;

    /* We should now have a DSI packet. */
    if(p[6] == 0x01) {
      nPacketLen = p[4] << 8 | p[5];
      p += 6;
      navRead_DSI(nav_dsi, p+1);
    }
    return 1;
  }
  return 0;
}

/* DSI is used for most angle stuff.
 * PCI is used for only non-seemless angle stuff
 */
static int32_t dvdnav_get_vobu(dvdnav_t *this, dsi_t *nav_dsi, pci_t *nav_pci, dvdnav_vobu_t *vobu) {
  uint32_t next;
  int32_t angle, num_angle;

  vobu->vobu_start = nav_dsi->dsi_gi.nv_pck_lbn; /* Absolute offset from start of disk */
  vobu->vobu_length = nav_dsi->dsi_gi.vobu_ea; /* Relative offset from vobu_start */

  /*
   * If we're not at the end of this cell, we can determine the next
   * VOBU to display using the VOBU_SRI information section of the
   * DSI.  Using this value correctly follows the current angle,
   * avoiding the doubled scenes in The Matrix, and makes our life
   * really happy.
   *
   * vobu_next is an offset value, 0x3fffffff = SRI_END_OF_CELL
   * DVDs are about 6 Gigs, which is only up to 0x300000 blocks
   * Should really assert if bit 31 != 1
   */

#if 0
  /* Old code -- may still be useful one day */
  if(nav_dsi->vobu_sri.next_vobu != SRI_END_OF_CELL ) {
    vobu->vobu_next = ( nav_dsi->vobu_sri.next_vobu & 0x3fffffff );
  } else {
    vobu->vobu_next = vobu->vobu_length;
  }
#else
  /* Relative offset from vobu_start */
  vobu->vobu_next = ( nav_dsi->vobu_sri.next_vobu & 0x3fffffff );
#endif

  vm_get_angle_info(this->vm, &angle, &num_angle);

  /* FIMXE: The angle reset doesn't work for some reason for the moment */
#if 0
  if((num_angle < angle) && (angle != 1)) {
    Log3(this, "angle ends!");

    /* This is to switch back to angle one when we
     * finish with angles. */
    dvdnav_angle_change(this, 1);
  }
#endif
  /* only use ILVU information if we are at the last vobunit in ILVU */
  /* otherwise we will miss nav packets from vobunits inbetween */
  if(num_angle != 0 && (nav_dsi->sml_pbi.category & DSI_ILVU_MASK) == (DSI_ILVU_BLOCK | DSI_ILVU_LAST)) {

    if((next = nav_pci->nsml_agli.nsml_agl_dsta[angle-1]) != 0) {
      if((next & 0x3fffffff) != 0) {
        if(next & 0x80000000)
          vobu->vobu_next = - (int32_t)(next & 0x3fffffff);
        else
          vobu->vobu_next = + (int32_t)(next & 0x3fffffff);
      }
    } else if((next = nav_dsi->sml_agli.data[angle-1].address) != 0) {
      vobu->vobu_length = nav_dsi->sml_pbi.ilvu_ea;

      if((next & 0x80000000) && (next != 0x7fffffff))
        vobu->vobu_next =  - (int32_t)(next & 0x3fffffff);
      else
        vobu->vobu_next =  + (int32_t)(next & 0x3fffffff);
    }
  }

  return 1;
}

/*
 * These are the main get_next_block function which actually get the media stream video and audio etc.
 *
 * There are two versions: The second one is using the zero-copy read ahead cache and therefore
 * hands out pointers targeting directly into the cache.
 * The first one uses a memcopy to fill this cache block into the application provided memory.
 * The benefit of this first one is that no special memory management is needed. The application is
 * the only one responsible of allocating and freeing the memory associated with the pointer.
 * The drawback is the additional memcopy.
 */

dvdnav_status_t dvdnav_get_next_block(dvdnav_t *this, uint8_t *buf,
                                      int32_t *event, int32_t *len) {
  unsigned char *block;
  dvdnav_status_t status;

  block = buf;
  status = dvdnav_get_next_cache_block(this, &block, event, len);
  if (status == DVDNAV_STATUS_OK && block != buf) {
    /* we received a block from the cache, copy it, so we can give it back */
    if (this->nav_only == 0) {
      memcpy(buf, block, DVD_VIDEO_LB_LEN);
    } else {
      *len = 0;
    }
#ifndef DVDNAV_NO_CACHE
    dvdnav_free_cache_block(this, block);
#endif
  }
  return status;
}

int64_t dvdnav_get_current_time(dvdnav_t *this) {
  int i;
  int64_t tm=0;
  dvd_state_t *state = &this->vm->state;

  if(! state->pgc)
    return 0;

  for(i=0; i<state->cellN-1; i++) {
    if(!
        (state->pgc->cell_playback[i].block_type == BLOCK_TYPE_ANGLE_BLOCK &&
         state->pgc->cell_playback[i].block_mode != BLOCK_MODE_FIRST_CELL)
    )
      tm += dvdnav_convert_time(&state->pgc->cell_playback[i].playback_time);
  }
  tm += this->cur_cell_time;

  return tm;
}

uint64_t dvdnav_get_absolute_time(dvdnav_t *self)
{
    dvd_state_t *state = &self->vm->state;

    return (((uint64_t)state->registers.time_counter) << 9);
}

uint32_t dvdnav_get_prm(dvdnav_t *self, uint8_t type, uint8_t reg)
{
    dvd_state_t *state = &self->vm->state;
    uint32_t r;

    switch (type)
    {
    case 0: r = state->registers.GPRM[reg]; break;
    case 1: r = state->registers.SPRM[reg]; break;
    case 2: r = state->registers.GPRM_mode; break;
    case 3: r = state->registers.SPRM_flags; break;
    default: r = 0; break;
    }
    return r;
}

static dvdnav_status_t dvdnav_check_vobu(dvdnav_t *this,uint32_t vobu)
{
    uint8_t buf[DVD_VIDEO_LB_LEN];
    uint8_t* data;
    pci_t pci;
    dsi_t dsi;

#ifndef DVDNAV_NO_CACHE
    data = buf;
    if (dvdnav_read_cache_block(this->cache, vobu, 1, &data) <= 0) data = NULL;
#else
    data = (uint8_t*) DVDReadBlock(this->file, vobu, buf);
#endif

    if (NULL == data) {
        printerr("Error reading NAV packet.");
        return DVDNAV_STATUS_ERR;
    }

    /* Decode nav into pci and dsi. */
    if (!dvdnav_decode_packet(this, data, &dsi, &pci)) {
        printerr("Expected NAV packet but none found.");
        return DVDNAV_STATUS_ERR;
    }

    if (pci.pci_gi.nv_pck_lbn != vobu) {
        printerr("Fake NAV packet.");
        return DVDNAV_STATUS_ERR;
    }

    return DVDNAV_STATUS_OK;
}

dvdnav_status_t dvdnav_get_next_cache_block(dvdnav_t *this, uint8_t **buf,
                                            int32_t *event, int32_t *len) {
  dvd_state_t *state;
  int32_t result;

  if(!this)
      return DVDNAV_STATUS_ERR;

  pthread_mutex_lock(&this->vm_lock);

  if(!this->started) {
    /* Start the VM */
    if (!vm_start(this->vm)) {
      printerr("Encrypted or faulty DVD");
      pthread_mutex_unlock(&this->vm_lock);
      return DVDNAV_STATUS_ERR;
    }
    this->started = 1;
  }

  if (!this->vm->state.pgc) {
      printerr("No current PGC.");
      pthread_mutex_unlock(&this->vm_lock);
      return DVDNAV_STATUS_ERR;
  }

  state = &(this->vm->state);
  (*event) = DVDNAV_NOP;
  (*len) = 0;

  /* Check the STOP flag */
  if(this->vm->stopped) {
    vm_stop(this->vm);
    (*event) = DVDNAV_STOP;
    this->started = 0;
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_OK;
  }

  vm_position_get(this->vm, &this->position_next);

#ifdef LOG_DEBUG
  Log3(this->vm, "POS-NEXT ");
  vm_position_print(this->vm, &this->position_next);
  Log3(this->vm, "POS-CUR  ");
  vm_position_print(this->vm, &this->position_current);
#endif

  /* did we hop? */
  if(this->position_current.hop_channel != this->position_next.hop_channel) {
    (*event) = DVDNAV_HOP_CHANNEL;
#ifdef LOG_DEBUG
    Log3(this->vm, "HOP_CHANNEL");
#endif
    if (this->position_next.hop_channel - this->position_current.hop_channel >= HOP_SEEK) {
      int32_t num_angles = 0, current;

      /* we seeked -> check for multiple angles */
      vm_get_angle_info(this->vm, &current, &num_angles);
      if (num_angles > 1) {
        int32_t block;
        /* we have to skip the first VOBU when seeking in a multiangle feature,
         * because it might belong to the wrong angle */
        block = this->position_next.cell_start + this->position_next.block;
#ifndef DVDNAV_NO_CACHE
        result = dvdnav_read_cache_block(this->cache, block, 1, buf);
#else
        *buf = (uint8_t*) DVDReadBlock(this->file, block, *buf);
        result = (NULL == *buf) ? -1 : DVD_VIDEO_LB_LEN;
#endif
        if(result <= 0) {
          printerr("Error reading NAV packet.");
          pthread_mutex_unlock(&this->vm_lock);
          return DVDNAV_STATUS_ERR;
        }
        /* Decode nav into pci and dsi. Then get next VOBU info. */
        if(!dvdnav_decode_packet(this, *buf, &this->dsi, &this->pci)) {
          printerr("Expected NAV packet but none found.");
          pthread_mutex_unlock(&this->vm_lock);
          return DVDNAV_STATUS_ERR;
        }
        dvdnav_get_vobu(this, &this->dsi, &this->pci, &this->vobu);
        /* skip to next, if there is a next */
        if (this->vobu.vobu_next != SRI_END_OF_CELL) {
          this->vobu.vobu_start += this->vobu.vobu_next;
          this->vobu.vobu_next   = 0;
        }
        /* update VM state */
        this->vm->state.blockN = this->vobu.vobu_start - this->position_next.cell_start;
      }
    }
    this->position_current.hop_channel = this->position_next.hop_channel;
    /* update VOBU info */
    this->vobu.vobu_start  = this->position_next.cell_start + this->position_next.block;
    this->vobu.vobu_next   = 0;
    /* Make blockN == vobu_length to do expected_nav */
    this->vobu.vobu_length = 0;
    this->vobu.blockN      = 0;
    this->sync_wait        = 0;
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_OK;
  }

  /* Check the HIGHLIGHT flag */
  if(this->position_current.button != this->position_next.button) {
    dvdnav_highlight_event_t *hevent = (dvdnav_highlight_event_t *)*buf;

    (*event) = DVDNAV_HIGHLIGHT;
#ifdef LOG_DEBUG
    Log3(this, "HIGHLIGHT");
#endif
    (*len) = sizeof(dvdnav_highlight_event_t);
    hevent->display = 1;
    hevent->buttonN = this->position_next.button;
    this->position_current.button = this->position_next.button;
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_OK;
  }

  /* Check the WAIT flag */
  if(this->sync_wait) {
    (*event) = DVDNAV_WAIT;
#ifdef LOG_DEBUG
    Log3(this, "WAIT");
#endif
    (*len) = 0;
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_OK;
  }

  /* Check to see if we need to change the currently opened VOB or open
   * a new one because we don't currently have an opened VOB. */
  if((this->file == NULL) ||
     (this->position_current.vts != this->position_next.vts) ||
     (this->position_current.domain != this->position_next.domain)) {
    dvd_read_domain_t domain;
    int32_t vtsN;
    dvdnav_vts_change_event_t *vts_event = (dvdnav_vts_change_event_t *)*buf;

    if(this->file) {
      DVDCloseFile(this->file);
      this->file = NULL;
    }

    vts_event->old_vtsN = this->position_current.vts;
    vts_event->old_domain = this->position_current.domain;

    /* Use the DOMAIN to find whether to open menu or title VOBs */
    switch(this->position_next.domain) {
    case DVD_DOMAIN_FirstPlay:
    case DVD_DOMAIN_VMGM:
      domain = DVD_READ_MENU_VOBS;
      vtsN = 0;
      break;
    case DVD_DOMAIN_VTSMenu:
      domain = DVD_READ_MENU_VOBS;
      vtsN = this->position_next.vts;
      break;
    case DVD_DOMAIN_VTSTitle:
      domain = DVD_READ_TITLE_VOBS;
      vtsN = this->position_next.vts;
      break;
    default:
      printerr("Unknown domain when changing VTS.");
      pthread_mutex_unlock(&this->vm_lock);
      return DVDNAV_STATUS_ERR;
    }

    this->position_current.vts = this->position_next.vts;
    this->position_current.domain = this->position_next.domain;
#ifndef DVDNAV_NO_CACHE
    dvdnav_read_cache_clear(this->cache);
#endif
    this->file = DVDOpenFile(vm_get_dvd_reader(this->vm), vtsN, domain);
    vts_event->new_vtsN = this->position_next.vts;
    vts_event->new_domain = this->position_next.domain;

    /* If couldn't open the file for some reason, moan */
    if(this->file == NULL) {
      printerrf("Error opening vtsN=%i, domain=%i.", vtsN, domain);
      pthread_mutex_unlock(&this->vm_lock);
      return DVDNAV_STATUS_ERR;
    }

    /* File opened successfully so return a VTS change event */
    (*event) = DVDNAV_VTS_CHANGE;
#ifdef LOG_DEBUG
    Log3(this, "VTS_CHANGE");
#endif
    (*len) = sizeof(dvdnav_vts_change_event_t);

    this->spu_clut_changed = 1;
    this->position_current.cell = -1; /* Force an update */
    this->position_current.spu_channel = -1; /* Force an update */
    this->position_current.audio_channel = -1; /* Force an update */;

    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_OK;
  }

  /* Check if the cell changed */
  if( (this->position_current.cell != this->position_next.cell) ||
      (this->position_current.cell_restart != this->position_next.cell_restart) ||
      (this->position_current.cell_start != this->position_next.cell_start) ) {
    dvdnav_cell_change_event_t *cell_event = (dvdnav_cell_change_event_t *)*buf;
    int32_t first_cell_nr, last_cell_nr, i;
    state = &this->vm->state;

    this->cur_cell_time = 0;
    (*event) = DVDNAV_CELL_CHANGE;
#ifdef LOG_DEBUG
    Log3(this, "CELL_CHANGE");
#endif
    (*len) = sizeof(dvdnav_cell_change_event_t);

    cell_event->cellN = state->cellN;
    cell_event->pgN   = state->pgN;
    cell_event->cell_length =
        (state->pgc->cell_playback[state->cellN - 1].last_sector + 1);

    cell_event->pg_length = 0;
    /* Find start cell of program. */
    first_cell_nr = state->pgc->program_map[state->pgN-1];
    /* Find end cell of program */
    if(state->pgN < state->pgc->nr_of_programs)
      last_cell_nr = state->pgc->program_map[state->pgN] - 1;
    else
      last_cell_nr = state->pgc->nr_of_cells;
    for (i = first_cell_nr; i <= last_cell_nr; i++)
      cell_event->pg_length +=
        (state->pgc->cell_playback[i - 1].last_sector + 1);

    cell_event->pgc_length = dvdnav_convert_time(&state->pgc->playback_time);

    cell_event->cell_start = 0;
    for (i = 1; i < state->cellN; i++)
      cell_event->cell_start +=
          (state->pgc->cell_playback[i - 1].last_sector + 1);

    cell_event->pg_start = 0;
    for (i = 1; i < state->pgc->program_map[state->pgN-1]; i++)
      cell_event->pg_start +=
        (state->pgc->cell_playback[i - 1].last_sector + 1);

    this->position_current.cell         = this->position_next.cell;
    this->position_current.cell_restart = this->position_next.cell_restart;
    this->position_current.cell_start   = this->position_next.cell_start;
    this->position_current.block        = this->position_next.block;

    /* vobu info is used for mid cell resumes */
    this->vobu.vobu_start               = this->position_next.cell_start + this->position_next.block;
    this->vobu.vobu_next                = 0;
    /* Make blockN == vobu_length to do expected_nav */
    this->vobu.vobu_length = 0;
    this->vobu.blockN      = 0;

    /* update the spu palette at least on PGC changes */
    this->spu_clut_changed = 1;
    this->position_current.spu_channel = -1; /* Force an update */
    this->position_current.audio_channel = -1; /* Force an update */

    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_OK;
  }

  /* has the CLUT changed? */
  if(this->spu_clut_changed) {
    (*event) = DVDNAV_SPU_CLUT_CHANGE;
#ifdef LOG_DEBUG
    Log3(this, "SPU_CLUT_CHANGE");
#endif
    (*len) = 16 * sizeof(uint32_t);
    memcpy(*buf, state->pgc->palette, sizeof(state->pgc->palette));
    this->spu_clut_changed = 0;
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_OK;
  }

  /* has the SPU channel changed? */
  if(this->position_current.spu_channel != this->position_next.spu_channel) {
    dvdnav_spu_stream_change_event_t *stream_change = (dvdnav_spu_stream_change_event_t *)*buf;

    (*event) = DVDNAV_SPU_STREAM_CHANGE;
#ifdef LOG_DEBUG
    Log3(this, "SPU_STREAM_CHANGE");
#endif
    (*len) = sizeof(dvdnav_spu_stream_change_event_t);
    stream_change->physical_wide      = vm_get_subp_active_stream(this->vm, 0);
    stream_change->physical_letterbox = vm_get_subp_active_stream(this->vm, 1);
    stream_change->physical_pan_scan  = vm_get_subp_active_stream(this->vm, 2);
    this->position_current.spu_channel = this->position_next.spu_channel;
#ifdef LOG_DEBUG
    Log3(this, "SPU_STREAM_CHANGE stream_id_wide=%d",stream_change->physical_wide);
    Log3(this, "SPU_STREAM_CHANGE stream_id_letterbox=%d",stream_change->physical_letterbox);
    Log3(this, "SPU_STREAM_CHANGE stream_id_pan_scan=%d",stream_change->physical_pan_scan);
    Log3(this, "SPU_STREAM_CHANGE returning DVDNAV_STATUS_OK");
#endif
    /* This is not really the right place to do this. FOSL_BTNN should set the register
     * at HLI_S_PTM rather than when we enter the SPU. As well we should activate FOAC_BTNN
     * at HLI_E_PTM
     */
    if (this->pci.hli.hl_gi.fosl_btnn != 0) {
      set_HL_BTN(this->vm, this->pci.hli.hl_gi.fosl_btnn);
    }

    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_OK;
  }

  /* has the audio channel changed? */
  if(this->position_current.audio_channel != this->position_next.audio_channel) {
    dvdnav_audio_stream_change_event_t *stream_change = (dvdnav_audio_stream_change_event_t *)*buf;

    (*event) = DVDNAV_AUDIO_STREAM_CHANGE;
#ifdef LOG_DEBUG
    Log3(this, "AUDIO_STREAM_CHANGE");
#endif
    (*len) = sizeof(dvdnav_audio_stream_change_event_t);
    stream_change->physical = vm_get_audio_active_stream( this->vm );
    stream_change->logical = this->position_next.audio_channel;
    this->position_current.audio_channel = this->position_next.audio_channel;
#ifdef LOG_DEBUG
    Log3(this, "AUDIO_STREAM_CHANGE stream_id=%d returning DVDNAV_STATUS_OK",stream_change->physical);
#endif
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_OK;
  }

  /* Check the STILLFRAME flag */
  if(this->position_current.still != 0) {
    dvdnav_still_event_t *still_event = (dvdnav_still_event_t *)*buf;

    (*event) = DVDNAV_STILL_FRAME;
#ifdef LOG_DEBUG
    Log3(this, "STILL_FRAME");
#endif
    (*len) = sizeof(dvdnav_still_event_t);
    still_event->length = this->position_current.still;
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_OK;
  }

  /* Have we reached the end of a VOBU? */
  if (this->vobu.blockN >= this->vobu.vobu_length) {

    /* Have we reached the end of a cell? */
    if(this->vobu.vobu_next == SRI_END_OF_CELL) {
      /* End of Cell from NAV DSI info */
#ifdef LOG_DEBUG
      Log3(this, "Still set to %x", this->position_next.still);
#endif
      this->position_current.still = this->position_next.still;

      /* we are about to leave a cell, so a lot of state changes could occur;
       * under certain conditions, the application should get in sync with us before this,
       * otherwise it might show stills or menus too shortly */
      if ((this->position_current.still || this->pci.hli.hl_gi.hli_ss) && !this->sync_wait_skip)
        this->sync_wait = 1;

      if(!this->position_current.still || this->skip_still ) {
        /* no active cell still -> get us to the next cell */
        vm_get_next_cell(this->vm);
        this->position_current.cell = -1;
        this->position_current.still = 0; /* still gets activated at end of cell */
        this->skip_still = 0;
        this->sync_wait_skip = 0;
      }
      /* handle related state changes in next iteration */
      (*event) = DVDNAV_NOP;
      (*len) = 0;
      pthread_mutex_unlock(&this->vm_lock);
      return DVDNAV_STATUS_OK;
    }

    /* at the start of the next VOBU -> expecting NAV packet */
#ifndef DVDNAV_NO_CACHE
    result = dvdnav_read_cache_block(this->cache, this->vobu.vobu_start + this->vobu.vobu_next, 1, buf);
#else
    *buf = (uint8_t*)DVDReadBlock(this->file, this->vobu.vobu_start + this->vobu.vobu_next, *buf);
    result = (NULL == *buf) ? -1 : DVD_VIDEO_LB_LEN;
#endif

    if(result <= 0) {
      printerr("Error reading NAV packet.");
      pthread_mutex_unlock(&this->vm_lock);
      return DVDNAV_STATUS_ERR;
    }
    /* Decode nav into pci and dsi. Then get next VOBU info. */
    if(!dvdnav_decode_packet(this, *buf, &this->dsi, &this->pci)) {
      printerr("Expected NAV packet but none found.");
      pthread_mutex_unlock(&this->vm_lock);
      return DVDNAV_STATUS_ERR;
    }
    /* We need to update the vm state->blockN with which VOBU we are in.
     * This is so RSM resumes to the VOBU level and not just the CELL level.
     */
    this->vm->state.blockN = this->vobu.vobu_start - this->position_current.cell_start;

    dvdnav_get_vobu(this, &this->dsi, &this->pci, &this->vobu);
    this->vobu.blockN = 0;

#ifndef DVDNAV_NO_CACHE
    /* Give the cache a hint about the size of next VOBU.
     * This improves pre-caching, because the VOBU will almost certainly be read entirely.
     */
    dvdnav_pre_cache_blocks(this->cache, this->vobu.vobu_start+1, this->vobu.vobu_length+1);
#endif

    /* release NAV menu filter, when we reach the same NAV packet again */
    if (this->last_cmd_nav_lbn == this->pci.pci_gi.nv_pck_lbn)
      this->last_cmd_nav_lbn = SRI_END_OF_CELL;

    /* Successfully got a NAV packet */
    (*event) = DVDNAV_NAV_PACKET;
#ifdef LOG_DEBUG
    Log3(this, "NAV_PACKET");
#endif
    (*len) = 2048;

    {
        uint32_t new_cell_time = (uint32_t)dvdnav_convert_time(&this->dsi.dsi_gi.c_eltm);
        if (0 == this->cur_cell_time) {
            if (new_cell_time < (2 * 90000)) {
                this->vm->state.registers.time_counter += (new_cell_time>>9);
            }
            if (0 == new_cell_time) {
                new_cell_time = 1;
            }
        } else {
            uint32_t cur_cell_time = this->cur_cell_time;
            if ((new_cell_time > cur_cell_time) &&
                ((new_cell_time - cur_cell_time) < (260*60*90000))) {
                this->vm->state.registers.time_counter += ((new_cell_time - cur_cell_time)>>9);
            }
        }
        this->cur_cell_time = new_cell_time;
        if (this->vobu.vobu_next < 0) {
            this->vm->state.registers.time_counter += 88;
        }

    }

    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_OK;
  }

  /* If we've got here, it must just be a normal block. */
  if(!this->file) {
    printerr("Attempting to read without opening file.");
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_ERR;
  }

  if (this->nav_only) {
      const pgc_t* pgc = this->vm->state.pgc;

      if ((NULL != pgc) && (0==pgc->pg_playback_mode) && (this->vm->state.domain == DVD_DOMAIN_VTSTitle)) {

          uint32_t fwda = 0;
          cell_playback_t* cell = &pgc->cell_playback[this->vm->state.cellN - 1];

          if ( (this->pci.pci_gi.nv_pck_lbn == this->vobu.vobu_start) &&
              (this->pci.hli.hl_gi.hli_ss != 0) )
          {
              fwda = 1;
          }

          if ((0 == fwda) && (0 == cell->playback_mode)) {
              if ((BLOCK_TYPE_NONE == cell->block_type) &&
                  (BLOCK_MODE_NOT_IN_BLOCK == cell->block_mode) &&
                  (0 == cell->interleaved) &&
                  (0 == cell->seamless_angle) &&
                  (cell->last_vobu_start_sector < cell->last_sector) &&
                  (cell->last_vobu_start_sector > cell->first_sector) &&
                  (cell->first_sector < cell->last_sector) &&
                  (this->vobu.vobu_start >= cell->first_sector) &&
                  (this->vobu.vobu_start < cell->last_vobu_start_sector))
              {
                  fwda = cell->last_vobu_start_sector - this->vobu.vobu_start;
                  if ((fwda < 0x1000) || (DVDNAV_STATUS_ERR == dvdnav_check_vobu(this, cell->last_vobu_start_sector))) {
                      fwda = 0;
                  }
              }

              if ((0 == fwda) && (this->dsi.dsi_gi.nv_pck_lbn == this->vobu.vobu_start)) {
                  vobu_sri_t* sri = &this->dsi.vobu_sri;
                  unsigned int i;

                  for (i = 0; i < 16; i++)
                  {
                      if ((sri->fwda[i] & 0x3fffffff) >= 0x20000) continue;

                      fwda = sri->fwda[i] & 0x3fffffff;
                      break;
                  }
              }
          }
          if ((fwda > 0x20) && (DVDNAV_STATUS_ERR != dvdnav_check_vobu(this, this->vobu.vobu_start + fwda))) {
              this->vobu.vobu_next = fwda;
          }
      }
      this->vobu.blockN = this->vobu.vobu_length;

      (*event) = DVDNAV_BLOCK_OK;
      (*len) = 0;
  } else {
    this->vobu.blockN++;

#ifndef DVDNAV_NO_CACHE
    result = dvdnav_read_cache_block(this->cache, this->vobu.vobu_start + this->vobu.blockN, 1, buf);
#else
    *buf = (uint8_t*)DVDReadBlock(this->file, this->vobu.vobu_start + this->vobu.blockN, *buf);
    result = (NULL == *buf) ? -1 : DVD_VIDEO_LB_LEN;
#endif

    if(result <= 0) {
      printerr("Error reading from DVD.");
      pthread_mutex_unlock(&this->vm_lock);
      return DVDNAV_STATUS_ERR;
    }
    (*event) = DVDNAV_BLOCK_OK;
    (*len) = 2048;
  }

  pthread_mutex_unlock(&this->vm_lock);
  return DVDNAV_STATUS_OK;
}

#ifndef DVDNAV_NO_TXTDT
dvdnav_status_t dvdnav_get_title_string(dvdnav_t *this, const char **title_str) {
  (*title_str) = this->vm->dvd_name;
  return DVDNAV_STATUS_OK;
}

dvdnav_status_t dvdnav_get_serial_string(dvdnav_t *this, const char **serial_str) {
  (*serial_str) = this->vm->dvd_serial;
  return DVDNAV_STATUS_OK;
}

char * dvdnav_get_volid_string(dvdnav_t *this) {
  if (!this || !this->vm || !this->vm->dvd) {
    printerr("Invalid state, vm or reader not available.");
    return NULL;
  }

  char *volid_str = malloc(33);
  if (volid_str == NULL) {
    printerr("Insufficient memory available.");
    return NULL;
  }

  if (DVDUDFVolumeInfo(this->vm->dvd, volid_str, 32, NULL, 0) == -1) {
    if (DVDISOVolumeInfo(this->vm->dvd, volid_str, 33, NULL, 0) == -1) {
      printerr("Failed to obtain volume id.");
      free(volid_str);
      return NULL;
    }
  }
  return volid_str;
}
#endif

uint8_t dvdnav_get_video_aspect(dvdnav_t *this) {
  uint8_t         retval;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1;
  }

  pthread_mutex_lock(&this->vm_lock);
  retval = (uint8_t)vm_get_video_aspect(this->vm);
  pthread_mutex_unlock(&this->vm_lock);

  return retval;
}
dvdnav_status_t dvdnav_get_video_resolution(dvdnav_t *this, uint32_t *width, uint32_t *height) {
  int w, h;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return DVDNAV_STATUS_ERR;
  }

  pthread_mutex_lock(&this->vm_lock);
  vm_get_video_res(this->vm, &w, &h);
  pthread_mutex_unlock(&this->vm_lock);

  *width  = w;
  *height = h;
  return DVDNAV_STATUS_OK;
}

uint8_t dvdnav_get_video_scale_permission(dvdnav_t *this) {
  uint8_t         retval;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1;
  }

  pthread_mutex_lock(&this->vm_lock);
  retval = (uint8_t)vm_get_video_scale_permission(this->vm);
  pthread_mutex_unlock(&this->vm_lock);

  return retval;
}

uint16_t dvdnav_audio_stream_to_lang(dvdnav_t *this, uint8_t stream) {
  audio_attr_t  attr;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1;
  }

  pthread_mutex_lock(&this->vm_lock);
  attr = vm_get_audio_attr(this->vm, stream);
  pthread_mutex_unlock(&this->vm_lock);

  if(attr.lang_type != 1)
    return 0xffff;

  return attr.lang_code;
}

uint16_t dvdnav_audio_stream_format(dvdnav_t *this, uint8_t stream) {
  audio_attr_t  attr;
  uint16_t format;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1; /* 0xffff */
  }

  pthread_mutex_lock(&this->vm_lock);
  attr = vm_get_audio_attr(this->vm, stream);
  pthread_mutex_unlock(&this->vm_lock);

  switch(attr.audio_format) {
  case 0:
    format = DVD_AUDIO_FORMAT_AC3;
    break;
  case 2: /* MPEG-1 or MPEG-2 without extension bitstream. */
  case 3: /* MPEG-2 with extension bitstream. */
    format = DVD_AUDIO_FORMAT_MPEG2_EXT;
    break;
  case 4:
    format = DVD_AUDIO_FORMAT_LPCM;
    break;
  case 6:
    format = DVD_AUDIO_FORMAT_DTS;
    break;
  case 7:
    format = DVD_AUDIO_FORMAT_SDDS;
    break;
  default:
    format = 0xffff;
    break;
  }

  return format;
}

uint16_t dvdnav_audio_stream_channels(dvdnav_t *this, uint8_t stream) {
  audio_attr_t  attr;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1; /* 0xffff */
  }

  pthread_mutex_lock(&this->vm_lock);
  attr = vm_get_audio_attr(this->vm, stream);
  pthread_mutex_unlock(&this->vm_lock);

  return attr.channels + 1;
}

uint16_t dvdnav_spu_stream_to_lang(dvdnav_t *this, uint8_t stream) {
  subp_attr_t  attr;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1;
  }

  pthread_mutex_lock(&this->vm_lock);
  attr = vm_get_subp_attr(this->vm, stream);
  pthread_mutex_unlock(&this->vm_lock);

  if(attr.type != 1)
    return 0xffff;

  return attr.lang_code;
}

int8_t dvdnav_get_audio_logical_stream(dvdnav_t *this, uint8_t audio_num) {
  int8_t       retval;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1;
  }

  pthread_mutex_lock(&this->vm_lock);
  if (!this->vm->state.pgc) {
    printerr("No current PGC.");
    pthread_mutex_unlock(&this->vm_lock);
    return -1;
  }
  retval = vm_get_audio_stream(this->vm, audio_num);
  pthread_mutex_unlock(&this->vm_lock);

  return retval;
}

int8_t dvdnav_get_number_of_streams(dvdnav_t *this, dvdnav_stream_type_t stream_type) {

  if (stream_type != DVD_SUBTITLE_STREAM && stream_type != DVD_AUDIO_STREAM) {
    printerr("Invalid provided stream type");
    return -1;
  }

  if (!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1;
  }

  pthread_mutex_lock(&this->vm_lock);
  if (!this->vm->state.pgc) {
    printerr("No current PGC.");
    pthread_mutex_unlock(&this->vm_lock);
    return -1;
  }

  if (this->vm->state.domain != DVD_DOMAIN_VTSTitle &&
      this->vm->state.domain != DVD_DOMAIN_VTSMenu)
  {
    printerr("Invalid domain provided");
    pthread_mutex_unlock(&this->vm_lock);
    return -1;
  }

  int8_t count = 0;
  switch (stream_type) {
  case DVD_SUBTITLE_STREAM:
    for (int i = 0; i < 32; i++)
    {
      if (this->vm->state.pgc->subp_control[i] & (1<<31))
        count++;
    }
    break;
  case DVD_AUDIO_STREAM:
    for (int i = 0; i < 8; i++)
    {
      if (this->vm->state.pgc->audio_control[i] & (1<<15))
        count++;
    }
    break;
  }
  pthread_mutex_unlock(&this->vm_lock);
  return count;
}

dvdnav_status_t dvdnav_get_audio_attr(dvdnav_t *this, uint8_t audio_num, audio_attr_t *audio_attr) {
  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return DVDNAV_STATUS_ERR;
  }
  pthread_mutex_lock(&this->vm_lock);
  if (!this->vm->state.pgc) {
    printerr("No current PGC.");
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_ERR;
  }
  *audio_attr=vm_get_audio_attr(this->vm, audio_num);
  pthread_mutex_unlock(&this->vm_lock);

  return DVDNAV_STATUS_OK;
}

int8_t dvdnav_get_spu_logical_stream(dvdnav_t *this, uint8_t subp_num) {
  int8_t       retval;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1;
  }

  pthread_mutex_lock(&this->vm_lock);
  if (!this->vm->state.pgc) {
    printerr("No current PGC.");
    pthread_mutex_unlock(&this->vm_lock);
    return -1;
  }
  retval = vm_get_subp_stream(this->vm, subp_num, 0);
  pthread_mutex_unlock(&this->vm_lock);

  return retval;
}

dvdnav_status_t dvdnav_get_spu_attr(dvdnav_t *this, uint8_t audio_num, subp_attr_t *subp_attr) {
  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return DVDNAV_STATUS_ERR;
  }
  pthread_mutex_lock(&this->vm_lock);
  if (!this->vm->state.pgc) {
    printerr("No current PGC.");
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_ERR;
  }
  *subp_attr=vm_get_subp_attr(this->vm, audio_num);
  pthread_mutex_unlock(&this->vm_lock);
  return DVDNAV_STATUS_OK;
}

int8_t dvdnav_get_active_audio_stream(dvdnav_t *this) {
  int8_t        retval;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1;
  }

  pthread_mutex_lock(&this->vm_lock);
  if (!this->vm->state.pgc) {
    printerr("No current PGC.");
    pthread_mutex_unlock(&this->vm_lock);
    return -1;
  }
  retval = vm_get_audio_active_stream(this->vm);
  pthread_mutex_unlock(&this->vm_lock);

  return retval;
}

int8_t dvdnav_get_active_spu_stream(dvdnav_t *this) {
  int8_t        retval;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1;
  }

  pthread_mutex_lock(&this->vm_lock);
  if (!this->vm->state.pgc) {
    printerr("No current PGC.");
    pthread_mutex_unlock(&this->vm_lock);
    return -1;
  }
  retval = vm_get_subp_active_stream(this->vm, 0);
  pthread_mutex_unlock(&this->vm_lock);

  return retval;
}

dvdnav_status_t dvdnav_set_active_stream(dvdnav_t *this, uint8_t stream_num, dvdnav_stream_type_t stream_type) {
   if (stream_type != DVD_SUBTITLE_STREAM && stream_type != DVD_AUDIO_STREAM) {
    printerr("Invalid provided stream type");
    return DVDNAV_STATUS_ERR;
  }

  if (!this->started) {
    printerr("Virtual DVD machine not started.");
    return DVDNAV_STATUS_ERR;
  }

  pthread_mutex_lock(&this->vm_lock);
  if (!this->vm->state.pgc) {
    printerr("No current PGC.");
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_ERR;
  }

  if (this->vm->state.domain != DVD_DOMAIN_VTSTitle &&
      this->vm->state.domain != DVD_DOMAIN_VTSMenu)
  {
    printerr("Invalid active domain");
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_ERR;
  }

  switch (stream_type) {
  case DVD_SUBTITLE_STREAM:
    if (stream_num >= 32 ||
        !(this->vm->state.pgc->subp_control[stream_num] & (1 << 31))) {
      printerr("Invalid stream index not allowed");
      pthread_mutex_unlock(&this->vm_lock);
      return DVDNAV_STATUS_ERR;
    }
    // set state without changing the current visibility
    this->vm->state.SPST_REG = stream_num | (this->vm->state.SPST_REG & 0x40);
    break;
  case DVD_AUDIO_STREAM:
    if (stream_num >= 8 ||
        !(this->vm->state.pgc->audio_control[stream_num] & (1 << 15))) {
      printerr("Invalid stream index not allowed");
      pthread_mutex_unlock(&this->vm_lock);
      return DVDNAV_STATUS_ERR;
    }
    this->vm->state.AST_REG = stream_num;
    break;
  }
  pthread_mutex_unlock(&this->vm_lock);
  return DVDNAV_STATUS_OK;
}

dvdnav_status_t dvdnav_toggle_spu_stream(dvdnav_t *this, uint8_t visibility) {
  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return DVDNAV_STATUS_ERR;
  }

  pthread_mutex_lock(&this->vm_lock);
  switch(visibility) {
  case 0: /* disable */
    this->vm->state.SPST_REG &= ~0x40;
    break;
  case 1:  /* enable */
    this->vm->state.SPST_REG |= 0x40;
    break;
  default:
    printerr("Invalid provided enabled_flag value");
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_ERR;
  }
  pthread_mutex_unlock(&this->vm_lock);
  return DVDNAV_STATUS_OK;
}

static int8_t dvdnav_is_domain(dvdnav_t *this, DVDDomain_t domain) {
  int8_t        retval;

  if(!this->started) {
    printerr("Virtual DVD machine not started.");
    return -1;
  }

  pthread_mutex_lock(&this->vm_lock);
  retval = (this->vm->state.domain == domain);
  pthread_mutex_unlock(&this->vm_lock);

  return retval;
}

/* First Play domain. (Menu) */
int8_t dvdnav_is_domain_fp(dvdnav_t *this) {
  return dvdnav_is_domain(this, DVD_DOMAIN_FirstPlay);
}
/* Video management Menu domain. (Menu) */
int8_t dvdnav_is_domain_vmgm(dvdnav_t *this) {
  return dvdnav_is_domain(this, DVD_DOMAIN_VMGM);
}
/* Video Title Menu domain (Menu) */
int8_t dvdnav_is_domain_vtsm(dvdnav_t *this) {
  return dvdnav_is_domain(this, DVD_DOMAIN_VTSMenu);
}
/* Video Title domain (playing movie). */
int8_t dvdnav_is_domain_vts(dvdnav_t *this) {
  return dvdnav_is_domain(this, DVD_DOMAIN_VTSTitle);
}

/* Generally delegate angle information handling to VM */
dvdnav_status_t dvdnav_angle_change(dvdnav_t *this, int32_t angle) {
  int32_t num, current;

  pthread_mutex_lock(&this->vm_lock);
  vm_get_angle_info(this->vm, &current, &num);
  /* Set angle SPRM if valid */
  if((angle > 0) && (angle <= num)) {
    this->vm->state.AGL_REG = angle;
  } else {
    printerr("Passed an invalid angle number.");
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_ERR;
  }
  pthread_mutex_unlock(&this->vm_lock);

  return DVDNAV_STATUS_OK;
}

dvdnav_status_t dvdnav_get_angle_info(dvdnav_t *this, int32_t *current_angle,
                                      int32_t *number_of_angles) {
  pthread_mutex_lock(&this->vm_lock);
  vm_get_angle_info(this->vm, current_angle, number_of_angles);
  pthread_mutex_unlock(&this->vm_lock);

  return DVDNAV_STATUS_OK;
}

dvdnav_status_t dvdnav_get_disk_region_mask(dvdnav_t *this, int32_t *region_mask) {
  pthread_mutex_lock(&this->vm_lock);
  if (!this->vm || !this->vm->vmgi || !this->vm->vmgi->vmgi_mat) {
    printerr("Bad VM state.");
    pthread_mutex_unlock(&this->vm_lock);
    return DVDNAV_STATUS_ERR;
  }

  (*region_mask) = ((this->vm->vmgi->vmgi_mat->vmg_category >> 16) & 0xff) ^ 0xff;
  pthread_mutex_unlock(&this->vm_lock);
  return DVDNAV_STATUS_OK;
}

pci_t* dvdnav_get_current_nav_pci(dvdnav_t *this) {
  if(!this) return 0;
  return &this->pci;
}

dsi_t* dvdnav_get_current_nav_dsi(dvdnav_t *this) {
  if(!this) return 0;
  return &this->dsi;
}

uint32_t dvdnav_get_next_still_flag(dvdnav_t *this) {
  if(!this) return -1;
  return this->position_next.still;
}

user_ops_t dvdnav_get_restrictions(dvdnav_t* this) {
  /*
   * user_ops_t is a structure of 32 bits.  We want to compute
   * the union of two of those bitfields so to make this quicker
   * than performing 32 ORs, we will access them as 32bits words.
   */
  union {
    user_ops_t ops_struct;
    uint32_t   ops_int;
  } ops, tmp;

  ops.ops_int = 0;

  if(!this || !this->started) {
    printerr("Virtual DVD machine not started.");
    return ops.ops_struct;
  }

  pthread_mutex_lock(&this->vm_lock);
  ops.ops_struct = this->pci.pci_gi.vobu_uop_ctl;

  if(this->vm && this->vm->state.pgc) {
    tmp.ops_struct = this->vm->state.pgc->prohibited_ops;
    ops.ops_int |= tmp.ops_int;
  }
  pthread_mutex_unlock(&this->vm_lock);

  return ops.ops_struct;
}

const char* dvdnav_version(void) {
    return DVDNAV_VERSION_STRING;
}
